<?php 
/**
* Extension of the Codeigniter string helper.
*
* @package direct-project-innovation-initiative
* @subpackage helpers
* @filesource
*/

load_libraries('Error', 'Validator');

/**
* Returns true if the first string is a substring of the second string.
* @param string
* @param string
* @return boolean
*/
function string_contains($needle, $haystack){		
	if(!validates_as('string', $needle)) return should_be('a string', $needle);
	if(!validates_as('string', $haystack)) return should_be('a string', $haystack);		
		
	if($needle=='' || $haystack == '') return false;			
	return (mb_strpos($haystack, $needle) !== FALSE); 
}

/**
* Returns true if the second string begins with the first string.
* @param string
* @param string
* @return boolean
*/
function string_begins_with($needle, $haystack){
	if(!validates_as('string', $needle)) return should_be('a string', $needle);
	if(!validates_as('string', $haystack)) return should_be('a string', $haystack);		

	if($needle=='' || $haystack == '') return false;	
	return (mb_strpos($haystack, $needle) === 0); 
}

/**
* Returns true if the second string ends with the first string.
* @param string
* @param string
* @return boolean
*/
function string_ends_with($needle, $haystack){
	if(!validates_as('string', $needle)) return should_be('a string', $needle);
	if(!validates_as('string', $haystack)) return should_be('a string', $haystack);		
	
	if($needle=='' || $haystack == '') return false;
	
	return (mb_strrpos($haystack, $needle) === (mb_strlen($haystack) - mb_strlen($needle))); 
}

/**
* Returns true if the last character in the string is punctuation (and not whitespace).
* @param string
* @return boolean
*/
function string_ends_with_punctuation($string){
	if(!validates_as('string', $string)) return should_be('a string', $string);
	if(empty($string)) return false;
	$last_char = mb_substr($string, mb_strlen($string)-1, 1);
	return !preg_match('/[\p{L}\p{N}\p{M}\s]/u', $last_char);
}

/**
* Like {@link strip_from_end()}, except that it strips from the beginning.
*
* @param string
* @param string
* @return string
*/
function strip_from_beginning($needle, $haystack){
	if(!validates_as('string', $needle)) return should_be('a string', $needle);
	if(!validates_as('string', $haystack)) return should_be('a string', $haystack);		
		
	if(string_begins_with($needle, $haystack))
		return mb_substr($haystack, mb_strlen($needle), mb_strlen($haystack));
	
	return $haystack;
}


/**
* Strip a string from the end of a string.
* PHP's native {@link rtrim()} treats the characters you give it as 'characters to choose from' for stripping from the end.
* In contrast, this treats $needle as a full string that ought to be removed from the end if it's there.
* For example:
* <code>
* rtrim('hello world', 'hdle'); //returns 'hello wor'
* strip_from_end('hello_world', hdle'); //returns 'hello world' because the string 'hdle' was not at the end of the string
* </code>
*
* @param string $needle
* @param string $haystack
* @return string $haystack with $needle stripped from the end
*/
function strip_from_end($needle, $haystack){
	if(!validates_as('string', $needle)) return should_be('a string', $needle);
	if(!validates_as('string', $haystack)) return should_be('a string', $haystack);		
	
	if(string_ends_with($needle, $haystack)) return mb_substr($haystack, 0, mb_strrpos($haystack, $needle));
	
	return $haystack;
}

/**
* Replace just the first instance of a string.
* PHP's native {@link str_replace()} will replace all instances of $needle.  Use this when you just want to replace the first instance of it.
*
* @param string $needle 
* @param string $replace_with
* @param string $haystack 
* @return string
*/
function replace_first_with($needle, $replace_with, $haystack){		
	foreach(array('needle', 'replace_with', 'haystack') as $var){
		if(!is_string($$var)) return should_be('a string', $$var);
	}
		
	if(!string_contains($needle, $haystack)) return $haystack;
		
	$needle_start = mb_strpos($haystack, $needle);
	return mb_substr($haystack, 0, $needle_start).$replace_with.mb_substr($haystack, $needle_start + mb_strlen($needle));	
}


/**
* Replace just the last instance of a string.
* PHP's native {@link str_replace()} will replace all instances of $needle.  Use this when you just want to replace the last instance of it.
*
* @param string $needle 
* @param string $replace_with
* @param string $haystack 
* @return string
*/
function replace_last_with($needle, $replace_with, $haystack){		
	foreach(array('needle', 'replace_with', 'haystack') as $var){
		if(!is_string($$var)) return should_be('a string', $$var);
	}
		
	if(!string_contains($needle, $haystack)) return $haystack;
		
	$needle_start = mb_strrpos($haystack, $needle);
	return mb_substr($haystack, 0, $needle_start).$replace_with.mb_substr($haystack, $needle_start + mb_strlen($needle));	
}


function number_length($number){
	if(!validates_as('numeric', $number)) return should_be('number', $number);
	if($number === 0) return 1;
	if($number < 1) $number = abs($number);
	if($number - floor($number) == 0){
		return mb_strlen(strip_from_end('.000000', sprintf("%f", $number)));        //seems to be the most reliable way to go about this -- check again if necessary
		#return floor( log($number, 10) ) + 1;
	}
	
	$integer_portion = floor( $number );
	$decimal_portion = $number - $integer_portion;
	$decimal_portion = strip_from_beginning('0.', rtrim(sprintf("%f", $decimal_portion), '0')); //MAY HAVE PRECISION PROBLEMS HERE.  grr, argh.
		
	return number_length($integer_portion) + number_length($decimal_portion);

}

//known limitation -- this will treat hyphenated words as separate words
function first_word($string){
	if(!validates_as('string', $string)) return should_be('string', $string);
	if(empty($string)) return '';
	#return current(preg_split('/[^a-zA-Z0-9]/',$string, -1, PREG_SPLIT_NO_EMPTY)); //this only works if you don't have any international chars
	return current(preg_split('/[^\p{L}\p{M}\p{N}-]/u',$string, -1, PREG_SPLIT_NO_EMPTY));
}

function last_word($string){
	if(!validates_as('string', $string)) return should_be('string', $string);
	if(empty($string)) return '';
	#return end(preg_split('/[^a-zA-Z0-9]/',$string, -1, PREG_SPLIT_NO_EMPTY)); //this only works if you don't have any international chars
	return end(preg_split('/[^\p{L}\p{M}\p{N}-]/u',$string, -1, PREG_SPLIT_NO_EMPTY));
}


/**
* Returns the number of bytes in this string.
* At one point (and still according to the PHP documentation), strlen() returned the number of bytes in a string, and 
* mb_strlen() returned the number of characters.  Now, this seems to be a matter of confusion because strlen() is sometimes 
*  overriden to use mb_strlen().  This method will be used as a wrapper for whichever method gives us accurate results,
* and should be kept up to date with the current system setup.
*
* See (@link http://stackoverflow.com/questions/2384260/how-do-i-find-the-number-of-bytes-within-utf-8-string-with-php} and 
* {@link http://www.php.net/manual/en/function.mb-strlen.php#77040}.
*
* @param string
* @return int
*/
function string_length_in_bytes($string){
	if(!validates_as('string', $string)) return should_be('string', $string);
	
	//as of 2014-04-18, this works the same as strlen(), but strlen() is going to stop working to meausure bytes in PHP6, so
	//we'll go with this in the hope that it saves us some heartache later. -- MG 2014-04-18
	if(function_exists('mb_strlen'))
		return mb_strlen($string, '8bit'); 
	return strlen($string);
}

//multibyte-safe version of PHP's native wordwrap: http://us2.php.net/manual/en/function.wordwrap.php#107570
function mb_wordwrap($string, $width=75, $break="\n", $cut=false)
{
  if($cut) {
    // Match anything 1 to $width chars long followed by whitespace or EOS,
    // otherwise match anything $width chars long
    $search = '/(.{1,'.$width.'})(?:\s|$)|(.{'.$width.'})/uS';
    $replace = '$1$2'.$break;
  } else {
    // Anchor the beginning of the pattern with a lookahead
    // to avoid crazy backtracking when words are longer than $width
    $search = '/(?=\s)(.{1,'.$width.'})(?:\s|$)/uS';
    $replace = '$1'.$break;
  }
  return preg_replace($search, $replace, $string);
}

function mb_str_split($string){
	if(!is_string($string)) return should_be('string', $string);
	return preg_split('/(?<!^)(?!$)/u', $string);
}

/* End of file ICARUS_string_helper.php */